#! /usr/bin/env python
#*******************************************************************************
# ALMA - Atacama Large Millimiter Array
# (c) Associated Universities Inc., 2009 
# 
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
#
# "@(#) $Id: ObsCalPadDelay.py 247921 2017-08-08 15:07:45Z ahirota $"

#
# forcing global imports is due to an OSS problem
#
global copy
import copy

global CCL
import CCL.Global

global Control
import Control

global ControlExceptionsImpl
import ControlExceptionsImpl

global Observation
import Observation.DelayCalTarget
import Observation.SSRTuning
import Observation.ObsCalBase


class ObsCalPadDelay(Observation.ObsCalBase.ObsCalBase):

    options = [
        Observation.ObsCalBase.scriptOption("band", int, 3),
        Observation.ObsCalBase.scriptOption("dumpDuration", float, 0.576),
        Observation.ObsCalBase.scriptOption("channelAverageDuration", float, 0.576),
        # ACA requires more than about 8.5 sec in FDM mode for 16-ant array
        Observation.ObsCalBase.scriptOption("integrationDuration", float, 9.216),
        Observation.ObsCalBase.scriptOption("tpIntegrationDuration", float, 0.016),
        Observation.ObsCalBase.scriptOption("ElLimit", str, "20 deg"),
        Observation.ObsCalBase.scriptOption("bbNames", str, "")
    ]

    def parseOptions(self):
        self.band                    = self.args.band
        self.dumpDuration            = self.args.dumpDuration
        self.channelAverageDuration  = self.args.channelAverageDuration
        self.integrationDuration     = self.args.integrationDuration
        self.tpIntegrationDuration   = self.args.tpIntegrationDuration
        self.elLimit                 = self.args.ElLimit
        bbNameStr                    = self.args.bbNames
        self.bbNames = None
        if bbNameStr is not None and bbNameStr != "":
            self.bbNames = []
            for s in bbNameStr.split(','):
                self.bbNames.append(s)

    def generateTunings(self):
        #
        # For B9/10 its useful to do an all-USB setup to avoid configing the
        # analysis into producing irrelevant LSB delays.
        #
        if self.band == 9:
            # can improve on this a lot by moving LOs around to minimise noise in the USB
            sameBBFreqs = True
            frequency = 668.0
            SBPref = "LSB"
        elif self.band == 10:
            sameBBFreqs = True
            frequency = 852.0
            SBPref = "LSB"
        else:
            sameBBFreqs = False
            frequency = Observation.SSRTuning.bandFreqs_delayMeasurement[self.band]
            SBPref = None
        corrType = self._array.getCorrelatorType()
        self._spectralSpec = self._tuningHelper.GenerateSpectralSpec(
                band = self.band,
                intent = "interferometry_continuum",
                frequency = frequency,
                bbNames = self.bbNames,
                SBPref = SBPref,
                sameBBFreqs = sameBBFreqs,
                corrType = corrType,
                corrMode = "FDM",
                bwd = 4,
                dualMode = True,
                dump = self.dumpDuration,
                channelAverage = self.channelAverageDuration,
                integration = self.integrationDuration)

    def checkCorrelator(self):
        # ICT-23071
        corrType = self._array.getCorrelatorType()
        if corrType == 'TPS':
            raise Exception("TPS does not accept FDM setup")

    def doDelayCal(self):
        src = self._srcPointFocus
        ss = self._spectralSpec
        subscanDuration = 30.0
        try:
            delayCal = Observation.DelayCalTarget.DelayCalTarget(src, ss)
            delayCal.setSubscanDuration(subscanDuration)
            delayCal.setIntegrationTime(1.0)
            self.logInfo('Executing DelayCal on ' + src.sourceName + '...')
            delayCal.execute(self._obsmode)
            self.logInfo('Completed DelayCal on ' + src.sourceName)
        except BaseException as ex:
            print(ex)
            msg = "Error executing cal survey scans on source %s" % src.sourceName
            self.logError(msg)
            self.closeExecution(ex)
            raise ex



obs = ObsCalPadDelay()
obs.parseOptions()
obs.checkAntennas()
obs.startPrepareForExecution()
try:
    obs.generateTunings()
    obs.checkCorrelator()
    obs.findPointFocusSource()
    obs.setTelCalRefAntennaList()
except BaseException as ex:
    obs.logException("Error in methods run during execution/obsmode startup", ex)
    obs.completePrepareForExecution()
    obs.closeExecution(ex)
    raise ex
obs.completePrepareForExecution()
obs.logInfo("Executing DelayCal...")
obs.doDelayCal()
obs.closeExecution()

