/*
 * Decompiled with CFR 0.152.
 */
package alma.common.horizonsparser;

import alma.acstime.Epoch;
import alma.entity.xmlbinding.valuetypes.AngularVelocityT;
import alma.entity.xmlbinding.valuetypes.DoubleWithUnitT;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;

public class HorizonParser {
    private static Map<HorizonKeyword, List<HorizonsUnit>> horizonKeywordUnitMap = new HashMap<HorizonKeyword, List<HorizonsUnit>>();
    Logger logger = null;
    String ephem = null;
    public static final double AU_TO_METERS = 1.495978707E11;
    private SingleParser dateParser;
    private DoubleParser positionParser;
    private DeltaParser deltaParser;
    private DeltaParser deldotParser;
    private String header;
    private String dateHeader = null;
    private String positionHeader = null;
    private String deltaHeader = null;
    private String deldotHeader = null;
    private int startLineOfTrimmedEphemeris;
    private int endLineOfTrimmedEphemeris;
    private List<String> beginningLines;
    private List<String> endingLines;
    private List<String> ephemeris;
    private List<String> trimmedEphemeris;
    private int numberOfHeadersFound = 0;
    public static final String ALMA_SITE_TOPOCENTRIC = "Atacama Large mm/sub-mm Array (ALMA) Center of Array";

    public HorizonParser(Logger devLog) {
        this.logger = devLog;
    }

    public HorizonParser(Logger logger, String ephem) throws IOException, UnableToParseEphemerisDataException {
        this.logger = logger;
        this.ephem = ephem;
        this.loadEphemeris();
    }

    public HorizonParser(Logger logger, String ephem, double duration, Calendar cal) throws UnableToParseEphemerisDataException, IOException {
        this.logger = logger;
        this.ephem = ephem;
        this.loadEphemeris();
        this.getTrimmedEphemeris(duration, cal);
        this.ephemeris = this.trimmedEphemeris;
    }

    private void loadEphemeris() throws IOException, UnableToParseEphemerisDataException {
        if (this.ephem != null) {
            this.readEphemerisString(this.ephem);
            this.findHeaders();
            if (this.dateHeader == null) {
                String msg = "HorizonResultParser: Could not determine Date Column(s)";
                this.logger.info("HorizonResultParser: Could not determine Date Column(s)");
                throw new UnableToParseEphemerisDataException("HorizonResultParser: Could not determine Date Column(s)");
            }
            if (this.positionHeader == null) {
                String msg = "HorizonResultParser: Could not determine Position Columns";
                this.logger.info("HorizonResultParser: Could not determine Position Columns");
                throw new UnableToParseEphemerisDataException("HorizonResultParser: Could not determine Position Columns");
            }
            if (this.deltaHeader == null) {
                String msg = "HorizonResultParser: Could not determine Delta Column";
                this.logger.info("HorizonResultParser: Could not determine Delta Column");
                throw new UnableToParseEphemerisDataException("HorizonResultParser: Could not determine Delta Column");
            }
            this.createPositionParser(this.positionHeader);
            this.createDeltaParser(this.deltaHeader);
            this.createDateParser(this.dateHeader);
            if (this.deldotHeader != null) {
                this.createDeldotParser(this.deldotHeader);
            }
        } else {
            String msg = "HorizonResultParser.loadEphemeris: fieldSource is null";
            this.logger.fine("HorizonResultParser.loadEphemeris: fieldSource is null");
            throw new UnableToParseEphemerisDataException("HorizonResultParser.loadEphemeris: fieldSource is null");
        }
    }

    private void getTrimmedEphemeris(double duration, Calendar cal) throws UnableToParseEphemerisDataException {
        if (duration == 0.0) {
            this.startLineOfTrimmedEphemeris = 0;
            this.endLineOfTrimmedEphemeris = this.ephemeris.size();
        } else {
            CoordPair pair = this.findEphemerisRange(duration, cal);
            this.startLineOfTrimmedEphemeris = (int)pair.value1;
            this.endLineOfTrimmedEphemeris = (int)pair.value2;
        }
        this.trimmedEphemeris = new ArrayList<String>();
        this.trimmedEphemeris.addAll(this.ephemeris.subList(this.startLineOfTrimmedEphemeris, this.endLineOfTrimmedEphemeris));
    }

    private CoordPair findEphemerisRange(double duration, Calendar cal) throws UnableToParseEphemerisDataException {
        Calendar now = cal;
        long fileStart = this.dateParser.parse(this.ephemeris.get(0));
        long fileStep = this.dateParser.parse(this.ephemeris.get(1)) - fileStart;
        long start = Math.max((now.getTimeInMillis() - fileStart) / fileStep - 5L, 0L);
        long end = Math.min(start + (long)((int)(duration * 1.0E7 / (double)fileStep)) + 10L, (long)this.ephemeris.size());
        CoordPair pair = new CoordPair(this, start, end);
        return pair;
    }

    public List<ParsedEphemerisLine> parseEphemeris() throws NumberFormatException, IOException, UnableToParseEphemerisDataException {
        ArrayList<ParsedEphemerisLine> parsedEphemerisString = new ArrayList<ParsedEphemerisLine>();
        List<String> ephemerisData = this.ephemeris;
        String rightAscensionFormatLine = null;
        for (String line : this.beginningLines) {
            if (!line.startsWith("RA format")) continue;
            rightAscensionFormatLine = line;
            break;
        }
        if (rightAscensionFormatLine == null) {
            throw new UnableToParseEphemerisDataException("Unable to determine RA format - cannot find tag \"RA format\"");
        }
        rightAscensionType raType = this.getRightAscensionFormat(rightAscensionFormatLine);
        for (String line : ephemerisData) {
            ParsedEphemerisLine singleLine = null;
            try {
                singleLine = this.parseEphemerisLine(line, raType, this.header);
            }
            catch (StringIndexOutOfBoundsException e) {
                throw new UnableToParseEphemerisDataException("Unable to parse " + line);
            }
            if (singleLine == null) {
                throw new UnableToParseEphemerisDataException("Parsing ephemeris string failed.");
            }
            parsedEphemerisString.add(singleLine);
        }
        return parsedEphemerisString;
    }

    private void readEphemerisString(String ephemerisString) throws IOException, UnableToParseEphemerisDataException {
        if (ephemerisString == null || ephemerisString.isEmpty()) {
            throw new UnableToParseEphemerisDataException("Ephemeris is empty");
        }
        BufferedReader rdr = new BufferedReader(new StringReader(ephemerisString));
        ArrayList<String> lines = new ArrayList<String>();
        String line = rdr.readLine();
        while (line != null) {
            lines.add(line);
            line = rdr.readLine();
        }
        rdr.close();
        int indexOfSOE = lines.indexOf("$$SOE");
        int indexOfEOE = lines.indexOf("$$EOE");
        if (indexOfEOE == -1 || indexOfSOE == -1) {
            throw new UnableToParseEphemerisDataException("Are you sure this is a valid horizons ephemeris file?");
        }
        this.beginningLines = lines.subList(0, indexOfSOE);
        this.header = (String)lines.get(indexOfSOE - 2);
        this.ephemeris = lines.subList(indexOfSOE + 1, indexOfEOE - 1);
        this.endingLines = lines.subList(indexOfEOE, lines.size());
    }

    private void findHeaders() {
        String[] title = this.header.trim().split(" +");
        this.numberOfHeadersFound = title.length;
        for (String s : title) {
            if (s.isEmpty()) continue;
            if (s.length() < 4) {
                String msg = "HorizonResultParser.findHeaders: Column name is too short. There might be something wrong with ephemeris columns.";
                this.logger.finer("HorizonResultParser.findHeaders: Column name is too short. There might be something wrong with ephemeris columns.");
                continue;
            }
            if (s.substring(0, 4).equals("Date") && this.dateHeader == null) {
                this.dateHeader = s;
                continue;
            }
            if (s.substring(0, 4).equals("R.A.") && this.positionHeader == null) {
                this.positionHeader = s;
                String[] parts = this.positionHeader.split("_");
                if (!parts[1].equals("(ICRF/J2000.0)") || !parts[1].equals("(J2000.0)") || !parts[3].equals("(ICRF/J2000.0)")) continue;
                this.positionHeader = null;
                this.logger.info("HorizonResultParser.findHeader: Something wrong with position input");
                continue;
            }
            if (s.equals("delta")) {
                this.deltaHeader = s;
                continue;
            }
            if (!s.equals("deldot")) continue;
            this.deldotHeader = s;
        }
    }

    private void createPositionParser(String positionHeader) throws UnableToParseEphemerisDataException {
        if (positionHeader.equals("R.A._(ICRF/J2000.0)_DEC")) {
            this.positionParser = new SexagesimalPositionParser();
        } else if (positionHeader.equals("R.A.___(ICRF/J2000.0)___DEC") || positionHeader.equals("R.A._________(ICRF)_________DEC") || positionHeader.equals("R.A._____(ICRF)_____DEC")) {
            this.positionParser = new SexagesimalPositionParser();
        } else if (positionHeader.equals("R.A._(J2000.0)_DEC.")) {
            this.positionParser = new DecimalPositionParser();
        } else {
            String msg = "HorizonResultParser.createPositionParser: Unrecognized position type: " + positionHeader;
            this.logger.warning(msg);
            throw new UnableToParseEphemerisDataException(msg);
        }
        int dataStart = this.header.indexOf(positionHeader);
        int dataEnd = dataStart + positionHeader.length();
        this.positionParser.setDataRange(dataStart, dataEnd);
    }

    private void createDeltaParser(String deltaHeader) {
        int dataEnd = this.header.indexOf(deltaHeader) + deltaHeader.length();
        int dataStart = dataEnd - 16;
        this.deltaParser = new DeltaParser();
        this.deltaParser.setDataRange(dataStart, dataEnd);
        if (Double.parseDouble(this.ephemeris.get(0).substring(dataStart, dataEnd)) > 100000.0) {
            this.deltaParser.setUnit("km");
        } else {
            this.deltaParser.setUnit("AU");
        }
    }

    private void createDeldotParser(String deldotHeader) {
        int dataEnd = this.header.indexOf(deldotHeader) + deldotHeader.length();
        int dataStart = dataEnd - 11;
        this.deldotParser = new DeltaParser();
        this.deldotParser.setDataRange(dataStart, dataEnd);
        this.deldotParser.setUnit("km");
    }

    private void createDateParser(String dateHeader) throws UnableToParseEphemerisDataException {
        if (dateHeader.equals("Date__(UT)__HR:MN")) {
            this.dateParser = new UTParserHM(this);
        } else if (dateHeader.equals("Date__(UT)__HR:MN:SS")) {
            this.dateParser = new UTParserHMS(this);
        } else if (dateHeader.equals("Date__(UT)__HR:MN:SC.fff")) {
            this.dateParser = new UTParserHMSF(this);
        } else if (dateHeader.equals("Date_________JDUT")) {
            this.dateParser = new JDParser(this);
        } else {
            String msg = "HorizonResultParser.createDateParser: Unrecognized date type: " + dateHeader;
            this.logger.warning(msg);
            throw new UnableToParseEphemerisDataException(msg);
        }
        int dataStart = this.header.indexOf(dateHeader);
        int dataEnd = dataStart + dateHeader.length();
        this.dateParser.setDataRange(dataStart, dataEnd);
    }

    private ParsedEphemerisLine parseEphemerisLine(String inputString, rightAscensionType ephemerisType, String header) throws NumberFormatException, UnableToParseEphemerisDataException {
        ParsedEphemerisLine parsedEphemerisLine = new ParsedEphemerisLine();
        String[] tokenArray = this.getTokens(inputString);
        if (tokenArray.length < 4) {
            throw new UnableToParseEphemerisDataException("Not enough columns in the ephemeris.");
        }
        long ts = this.dateParser.parse(inputString);
        Calendar cal = Calendar.getInstance();
        cal.setTimeZone(TimeZone.getTimeZone("UTC"));
        cal.setTimeInMillis(ts);
        cal.getTime();
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.ssss");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        parsedEphemerisLine.timeStamp = cal;
        parsedEphemerisLine.rightAscension = this.positionParser.parse((String)inputString).value1;
        parsedEphemerisLine.declination = this.positionParser.parse((String)inputString).value2;
        parsedEphemerisLine.delta = this.deltaParser.parse((String)inputString).value1;
        if (this.deldotHeader != null) {
            parsedEphemerisLine.deltaDot = this.deldotParser.parse((String)inputString).value1;
        }
        return parsedEphemerisLine;
    }

    private rightAscensionType getRightAscensionFormat(String header) throws UnableToParseEphemerisDataException {
        String[] tokenArray = this.getTokens(header);
        if (tokenArray[3].contains("HMS")) {
            return rightAscensionType.Sexagesimal;
        }
        if (tokenArray[3].contains("DEG")) {
            return rightAscensionType.Decimal;
        }
        String msg = "Ephemeris R.A. is not in a recognizable format: should be either HMS or Deg";
        this.logger.info("Ephemeris R.A. is not in a recognizable format: should be either HMS or Deg");
        throw new UnableToParseEphemerisDataException("Ephemeris R.A. is not in a recognizable format: should be either HMS or Deg");
    }

    public String getSourceName(String sourceEphemeris) throws UnableToParseEphemerisDataException {
        String sourceName = "None";
        try {
            BufferedReader rdr = new BufferedReader(new StringReader(sourceEphemeris));
            String line = rdr.readLine();
            while (line != null) {
                if (line.length() > 16 && line.substring(0, 16).equals("Target body name")) {
                    StringTokenizer st = new StringTokenizer(line);
                    String[] tokenArray = new String[st.countTokens()];
                    int i = 0;
                    while (st.hasMoreTokens()) {
                        tokenArray[i] = st.nextToken();
                        ++i;
                    }
                    if (tokenArray.length > 3) {
                        sourceName = tokenArray[3];
                    }
                    try {
                        Integer.parseInt(sourceName);
                        sourceName = tokenArray.length > 4 ? tokenArray[4] : "None";
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                }
                line = rdr.readLine();
            }
            rdr.close();
        }
        catch (Exception ex) {
            throw new UnableToParseEphemerisDataException(ex.getMessage());
        }
        return sourceName;
    }

    private String[] getTokens(String inputString) {
        StringTokenizer st = new StringTokenizer(inputString);
        String[] tokenArray = new String[st.countTokens()];
        int i = 0;
        while (st.hasMoreTokens()) {
            tokenArray[i] = st.nextToken();
            ++i;
        }
        return tokenArray;
    }

    public double convertRAtoRad(String rightAscension) throws NumberFormatException, UnableToParseEphemerisDataException {
        if (rightAscension.length() < 10) {
            throw new UnableToParseEphemerisDataException("Right Ascension string is too short.");
        }
        try {
            double RAinDeg = HorizonParser.HHMMSSToDeg(rightAscension);
            double RAinRad = RAinDeg * Math.PI / 180.0;
            return RAinRad;
        }
        catch (Exception e) {
            String msg = "Unable to convert right ascension from HHMMSS to deg.";
            this.logger.severe("Unable to convert right ascension from HHMMSS to deg.");
            throw new UnableToParseEphemerisDataException("Unable to convert right ascension from HHMMSS to deg.");
        }
    }

    public double convertDecToRad(String declination) throws NumberFormatException, UnableToParseEphemerisDataException {
        if (((String)declination).length() < 10) {
            throw new UnableToParseEphemerisDataException("Declination string is too short");
        }
        if (((String)declination).length() <= 11) {
            while (((String)declination).length() < 13) {
                declination = (String)declination + "0";
            }
        }
        try {
            double DecInDeg = HorizonParser.DDMMSSToDeg((String)declination);
            double DecInRad = DecInDeg * Math.PI / 180.0;
            return DecInRad;
        }
        catch (Exception e) {
            String msg = "Unable to convert declination from DDMMSS to deg.";
            this.logger.severe("Unable to convert declination from DDMMSS to deg.");
            throw new UnableToParseEphemerisDataException("Unable to convert declination from DDMMSS to deg.");
        }
    }

    public static double convertAngleToRadians(DoubleWithUnitT angle) {
        String unit = angle.getUnit();
        double value = angle.getContent();
        double convFactor = 1.0;
        if (unit.equals("deg")) {
            convFactor = Math.PI / 180;
        } else if (unit.equals("arcmin")) {
            convFactor = 2.908882086657216E-4;
        } else if (unit.equals("arcsec")) {
            convFactor = 4.84813681109536E-6;
        } else if (unit.equals("mas")) {
            convFactor = 4.8481368110953594E-9;
        }
        return value * convFactor;
    }

    public static double convertAngularVelocityToRadiansPerSecond(AngularVelocityT angvel) {
        String unit = angvel.getUnit();
        double value = angvel.getContent();
        double convFactor = 1.0;
        if (unit.equals("arcsec/s")) {
            convFactor = 4.84813681109536E-6;
        } else if (unit.equals("arcmin/s")) {
            convFactor = 2.908882086657216E-4;
        } else if (unit.equals("deg/s")) {
            convFactor = Math.PI / 180;
        } else if (unit.equals("mas/yr")) {
            convFactor = 1.5362818500441604E-16;
        }
        return value * convFactor;
    }

    private static double HHMMSSToDeg(String value) {
        String[] parts = value.split(":");
        int hours = Integer.parseInt(parts[0]);
        int minutes = Integer.parseInt(parts[1]);
        String[] secs = parts[2].split("\\.");
        int seconds = Integer.parseInt(secs[0]);
        Object us = secs[1];
        for (int j = secs[1].length(); j < 6; ++j) {
            us = (String)us + "0";
        }
        int milliseconds = Integer.parseInt(((String)us).substring(0, 3));
        int microseconds = Integer.parseInt(((String)us).substring(3, 6));
        double deg = (double)hours + (double)minutes / 60.0 + (double)seconds / 3600.0 + (double)milliseconds / 3600000.0 + (double)microseconds / 3.6E9;
        return deg * 15.0;
    }

    private static double DDMMSSToDeg(String value) {
        String[] parts = value.split(":");
        int degrees = Integer.parseInt(parts[0]);
        int minutes = Integer.parseInt(parts[1]);
        String[] secs = parts[2].split("\\.");
        int seconds = Integer.parseInt(secs[0]);
        Object us = secs[1];
        for (int j = secs[1].length(); j < 6; ++j) {
            us = (String)us + "0";
        }
        int milliseconds = Integer.parseInt(((String)us).substring(0, 3));
        int microseconds = Integer.parseInt(((String)us).substring(3, 6));
        int sign = 1;
        if (parts[0].substring(0, 1).equals("-")) {
            sign = -1;
            degrees = Math.abs(degrees);
        }
        double deg = (double)sign * ((double)degrees + (double)minutes / 60.0 + (double)seconds / 3600.0 + (double)milliseconds / 3600000.0 + (double)microseconds / 3.6E9);
        return deg;
    }

    public HorizonsValueUnitPair getStepSize(String ephemeris) throws UnableToParseEphemerisDataException, IOException {
        if (StringUtils.isBlank((String)ephemeris)) {
            throw new UnableToParseEphemerisDataException("Ephemeris has not been specified.");
        }
        BufferedReader rdr = new BufferedReader(new StringReader(ephemeris));
        Pattern pattern = Pattern.compile("^Step-size[ ]+:[ ](\\d+)[ ](calendar year|arcsec[ ]\\(VARIABLE TIME\\)|steps|minutes|calendar month)$", 2);
        String line = rdr.readLine();
        while (line != null) {
            Matcher matcher = pattern.matcher(line);
            if (matcher.find()) {
                if (matcher.groupCount() != 2) {
                    throw new UnableToParseEphemerisDataException("Unable to decode Step-size keyword: expecting 2 groupings in matched string");
                }
                HorizonsUnit stepSizeUnits = null;
                try {
                    stepSizeUnits = this.getHorizonUnit(HorizonKeyword.STEPSIZE, matcher.group(2));
                }
                catch (IllegalArgumentException e) {
                    throw new UnableToParseEphemerisDataException(e.getMessage());
                }
                assert (stepSizeUnits != null);
                return new HorizonsValueUnitPair(this, Double.parseDouble(matcher.group(1)), stepSizeUnits);
            }
            line = rdr.readLine();
        }
        rdr.close();
        throw new UnableToParseEphemerisDataException("Unable to locate Step-size keyword construction");
    }

    private HorizonsUnit getHorizonUnit(HorizonKeyword keyword, String unit) throws UnableToParseEphemerisDataException {
        if (keyword == null) {
            throw new NullPointerException("Illegal argument - the argument keyword cannot be null");
        }
        if (StringUtils.isBlank((String)unit)) {
            throw new IllegalArgumentException("unit string should not be empty");
        }
        if (!horizonKeywordUnitMap.containsKey((Object)keyword)) {
            throw new IllegalArgumentException(String.format("Unrecognised horizons keyword %s in map", new Object[]{keyword}));
        }
        HorizonsUnit horizonsUnit = HorizonsUnit.getEnum(unit);
        for (HorizonsUnit currentUnit : horizonKeywordUnitMap.get((Object)keyword)) {
            if (!horizonsUnit.equals((Object)currentUnit)) continue;
            return currentUnit;
        }
        throw new UnableToParseEphemerisDataException(String.format("Unrecognised unit string %s specified for keyword %s", new Object[]{unit, keyword}));
    }

    public final int getNumberOfHeadersFound() {
        return this.numberOfHeadersFound;
    }

    public String getCenterSiteName(String ephemeris) throws UnableToParseEphemerisDataException, IOException {
        if (StringUtils.isBlank((String)ephemeris)) {
            throw new UnableToParseEphemerisDataException("Ephemeris has not been specified.");
        }
        BufferedReader rdr = new BufferedReader(new StringReader(ephemeris));
        Pattern pattern = Pattern.compile("Center-site[ ]*name:[ ]*(.*)");
        String line = rdr.readLine();
        while (line != null) {
            Matcher matcher = pattern.matcher(line);
            if (matcher.find()) {
                rdr.close();
                return matcher.group(1).trim();
            }
            line = rdr.readLine();
        }
        rdr.close();
        throw new UnableToParseEphemerisDataException("Unable to locate centre-site name");
    }

    static {
        horizonKeywordUnitMap.put(HorizonKeyword.STEPSIZE, Arrays.asList(HorizonsUnit.CALENDARYEAR, HorizonsUnit.ARCSEC, HorizonsUnit.STEPS, HorizonsUnit.MINUTES, HorizonsUnit.CALENDARMONTH));
    }

    public static class UnableToParseEphemerisDataException
    extends Exception {
        public UnableToParseEphemerisDataException(String msg) {
            super(msg);
        }
    }

    public class CoordPair {
        public double value1;
        public double value2;

        public CoordPair(HorizonParser this$0) {
        }

        public CoordPair(HorizonParser this$0, double value1, double value2) {
            this.value1 = value1;
            this.value2 = value2;
        }
    }

    public abstract class SingleParser {
        public int startIndex;
        public int endIndex;

        public SingleParser(HorizonParser this$0) {
        }

        public void setDataRange(int startIndex, int endIndex) {
            this.startIndex = startIndex;
            this.endIndex = endIndex;
        }

        public abstract long parse(String var1) throws UnableToParseEphemerisDataException;
    }

    public static enum rightAscensionType {
        Sexagesimal,
        Decimal;

    }

    public class ParsedEphemerisLine {
        public Calendar timeStamp;
        public Epoch epoch;
        public String code;
        public double rightAscension;
        public double declination;
        public double delta;
        public double deltaDot;

        public ParsedEphemerisLine() {
        }

        public ParsedEphemerisLine(Calendar timeStamp, String code, double rightAscension, double declination, double delta, double deltaDot) {
            this.timeStamp = timeStamp;
            this.code = code;
            this.rightAscension = rightAscension;
            this.declination = declination;
            this.delta = delta;
            this.deltaDot = deltaDot;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + this.getOuterType().hashCode();
            result = 31 * result + (this.timeStamp == null ? 0 : this.timeStamp.hashCode());
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            ParsedEphemerisLine other = (ParsedEphemerisLine)obj;
            if (!this.getOuterType().equals(other.getOuterType())) {
                return false;
            }
            return !(this.timeStamp == null ? other.timeStamp != null : !this.timeStamp.equals(other.timeStamp));
        }

        private HorizonParser getOuterType() {
            return HorizonParser.this;
        }
    }

    public class SexagesimalPositionParser
    extends DoubleParser {
        public SexagesimalPositionParser() {
            super(HorizonParser.this);
        }

        @Override
        public CoordPair parse(String line) throws NumberFormatException, UnableToParseEphemerisDataException {
            String splitLine = line.substring(this.startIndex, this.endIndex);
            String[] tokenArray = HorizonParser.this.getTokens(splitLine);
            String rightAscensionString = tokenArray[0] + ":" + tokenArray[1] + ":" + tokenArray[2];
            String declinationString = tokenArray[3] + ":" + tokenArray[4] + ":" + tokenArray[5];
            double ra = HorizonParser.this.convertRAtoRad(rightAscensionString);
            double dec = HorizonParser.this.convertDecToRad(declinationString);
            CoordPair pair = new CoordPair(HorizonParser.this, ra, dec);
            return pair;
        }
    }

    public abstract class DoubleParser {
        public int startIndex;
        public int endIndex;

        public DoubleParser(HorizonParser this$0) {
        }

        public void setDataRange(int startIndex, int endIndex) {
            this.startIndex = startIndex;
            this.endIndex = endIndex;
        }

        public abstract CoordPair parse(String var1) throws NumberFormatException, UnableToParseEphemerisDataException;
    }

    public class DecimalPositionParser
    extends DoubleParser {
        public DecimalPositionParser() {
            super(HorizonParser.this);
        }

        @Override
        public CoordPair parse(String line) {
            String splitLine = line.substring(this.startIndex, this.endIndex);
            String[] tokenArray = HorizonParser.this.getTokens(splitLine);
            String rightAscensionString = tokenArray[0];
            String declinationString = tokenArray[1];
            double ra = Double.parseDouble(rightAscensionString) * Math.PI / 180.0;
            double dec = Double.parseDouble(declinationString) * Math.PI / 180.0;
            CoordPair pair = new CoordPair(HorizonParser.this, ra, dec);
            return pair;
        }
    }

    public class DeltaParser
    extends DoubleParser {
        String unit;

        public DeltaParser() {
            super(HorizonParser.this);
        }

        public void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public CoordPair parse(String line) {
            String splitLine = line.substring(this.startIndex, this.endIndex);
            double factor = 1000.0;
            if (this.unit.equals("km")) {
                factor = 1000.0;
            }
            if (this.unit.equals("AU")) {
                factor = 1.495978707E11;
            }
            double delta = Double.parseDouble(splitLine) * factor;
            CoordPair pair = new CoordPair(HorizonParser.this, delta, 0.0);
            return pair;
        }
    }

    public class UTParserHM
    extends UTParserBase {
        public UTParserHM(HorizonParser this$0) {
            super(this$0, new SimpleDateFormat("yyyy-MMM-dd HH:mm"));
        }
    }

    public class UTParserHMS
    extends UTParserBase {
        public UTParserHMS(HorizonParser this$0) {
            super(this$0, new SimpleDateFormat("yyyy-MMM-dd HH:mm:ss"));
        }
    }

    public class UTParserHMSF
    extends UTParserBase {
        public UTParserHMSF(HorizonParser this$0) {
            super(this$0, new SimpleDateFormat("yyyy-MMM-dd HH:mm:ss"));
        }

        @Override
        public void setDataRange(int startIndex, int endIndex) {
            this.startIndex = startIndex;
            this.endIndex = endIndex - 4;
        }

        @Override
        public long parse(String line) throws UnableToParseEphemerisDataException {
            long p = super.parse(line);
            double d = Integer.parseInt(line.substring(this.startIndex, this.endIndex + 4)) * 1000;
            return p + new Double(d).longValue();
        }
    }

    public class JDParser
    extends SingleParser {
        double jdOffset = 2299160.5;

        public JDParser(HorizonParser this$0) {
            super(this$0);
        }

        @Override
        public long parse(String line) {
            float jd = Float.parseFloat(line.substring(this.startIndex, this.endIndex));
            double value = ((double)jd - this.jdOffset) * 8.64E11;
            return new Double(value).longValue();
        }
    }

    public static enum HorizonKeyword {
        STEPSIZE,
        ELEVATIONCUTOFF;

    }

    public static enum HorizonsUnit {
        CALENDARYEAR("calendar year"),
        ARCSEC("arcsec (VARIABLE TIME)"),
        STEPS("steps"),
        MINUTES("minutes"),
        CALENDARMONTH("calendar month");

        String unitString = null;

        private HorizonsUnit(String unitString) {
            this.unitString = unitString;
        }

        public String getUnitString() {
            return this.unitString;
        }

        public static HorizonsUnit getEnum(String unit) {
            if (StringUtils.isEmpty((String)unit)) {
                throw new IllegalArgumentException("The unit string is empty");
            }
            for (HorizonsUnit stepSizeUnit : HorizonsUnit.values()) {
                if (!StringUtils.equalsIgnoreCase((String)stepSizeUnit.getUnitString(), (String)unit.trim())) continue;
                return stepSizeUnit;
            }
            throw new IllegalArgumentException(String.format("Unable to find matching enum for %s", unit));
        }
    }

    public class HorizonsValueUnitPair {
        double value;
        HorizonsUnit units;

        public HorizonsValueUnitPair(HorizonParser this$0, Double value, HorizonsUnit units) {
            if (value == null) {
                throw new NullPointerException("Illegal argument. The argument cannot be null: value");
            }
            if (units == null) {
                throw new NullPointerException("Illegal argument. The argument cannot be null: units");
            }
            this.value = value;
            this.units = units;
        }

        public double getValue() {
            return this.value;
        }

        public HorizonsUnit getUnits() {
            return this.units;
        }
    }

    public class UTParserBase
    extends SingleParser {
        SimpleDateFormat format;

        public UTParserBase(HorizonParser this$0, SimpleDateFormat format) {
            super(this$0);
            this.format = format;
        }

        @Override
        public long parse(String line) throws UnableToParseEphemerisDataException {
            String splitLine = line.substring(this.startIndex, this.endIndex);
            Date date = null;
            Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
            try {
                date = this.format.parse(splitLine);
            }
            catch (ParseException e) {
                throw new UnableToParseEphemerisDataException(e.getMessage());
            }
            cal.setTime(date);
            long value = cal.getTimeInMillis();
            return value;
        }
    }
}

